package ch.ethz.infsec.dresdenocltools;

import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.util.zip.ZipException;
import java.util.zip.ZipFile;

import tudresden.ocl20.codegen.java.JavaClassCodeGenerator;
import tudresden.ocl20.core.parser.OCL20CLI;

public class ConstraintsToJava {

	private boolean verbose = false;

	private boolean help = false;

	private File outputfile = null;

	private File model = null;

	private File constraints = null;

	/**
	 * Parse command line arguments.
	 * 
	 * @param args
	 *            contains the arguments.
	 */
	private void parseOptions(String[] args) {
		for (int i = 0; i < args.length; i++) {
			if (args[i].equals("--help")) {
				help = true;
			} else if (args[i].equals("--verbose")) {
				verbose = true;
			} else if (args[i].equals("--outfile")) {
				if (i + 1 < args.length) {
					outputfile = new File(args[i + 1]);
					i++;
				} else {
					throw new IllegalArgumentException(
							"--outfile needs an argument!");
				}
			} else {
				if (i == args.length - 2) {
					try {
						model = getModelFile(args[i]);
					} catch (Exception e) {
						e.printStackTrace();
						throw new IllegalArgumentException(
								"Error while opening model file", e);
					}
				} else if (i == args.length - 1) {
					constraints = new File(args[i]);
				} else {
					throw new IllegalArgumentException("Unknown option "
							+ args[i]);
				}
			}
		}
	}

	/**
	 * Get the model file. Extract if from a zip file if the file ends with
	 * .zargo and therefor should be an ArgoUML model.
	 * 
	 * @param file
	 *            containing the model.
	 * @return
	 * @throws ZipException
	 *             if opening the zip fails.
	 * @throws IOException
	 *             if opening the zip failes.
	 */
	private File getModelFile(String file) throws ZipException, IOException {
		File mfile = new File(file);
		if (file.endsWith(".zargo") && mfile.exists()) {
			String basename = mfile.getName().substring(0, file.length() - 6);
			ZipFile zipFile = new ZipFile(mfile);
			File temp = File.createTempFile(basename, ".xmi");
			temp.deleteOnExit();
			OCL20CLI.copyInputStream(zipFile.getInputStream(zipFile
					.getEntry(basename + ".xmi")), new BufferedOutputStream(
					new FileOutputStream(temp)));
			zipFile.close();
			return temp;
		} else {
			return mfile;
		}
	}

	/**
	 * Print some usage information.
	 */
	private void usage() {
		System.err.println("Usage: "
				+ ConstraintsToJava.class.getCanonicalName()
				+ " [--help] [--outfile <out>] <model.xmi> <constraints.ocl>");
	}

	/**
	 * Convert inputs to Java code.
	 * 
	 * @param args
	 *            contains the command line arguments.
	 */
	public ConstraintsToJava(String[] args) {
		parseOptions(args);
		if (help || model == null || constraints == null) {
			usage();
		} else {
			if (model.exists() && model.isFile()) {
				if (constraints.exists() && constraints.isFile()) {
					System.out.println("Generating code from model "
							+ model.getPath() + " and constraints "
							+ constraints.getPath());
					JavaClassCodeGenerator jccg = new JavaClassCodeGenerator(
							model.getPath(), constraints.getPath(), true);
					String code = jccg.generateCodeString();
					if (outputfile == null) {
						System.out.println(code);
					} else {
						try {
							writeCodeToFile(code);
						} catch (IOException e) {
							System.err.println("Could not write to outfile "
									+ outputfile + ".");
							e.printStackTrace();
						}
					}
				} else {
					System.err.println(constraints.getAbsolutePath()
							+ " does not exist or is not a file.");
				}
			} else {
				System.err.println(model.getAbsolutePath()
						+ " does not exist or is not a file.");
			}
		}
	}

	/**
	 * Writes a string to the outputfile.
	 * 
	 * @param code
	 *            contains the string.
	 * @throws IOException
	 *             if writing failed.
	 */
	private void writeCodeToFile(String code) throws IOException {
		OutputStream out = new FileOutputStream(outputfile);
		out.write(code.getBytes());
		out.close();
	}

	/**
	 * @param args
	 *            with arguments.
	 */
	public static void main(String[] args) {
		new ConstraintsToJava(args);
	}

}
