/*
 *              __  ____________        ____         __    
 *             / / / /_  __/ __/ ____  / __/______ _/ /__ _
 *            / /_/ / / / _\ \  /___/ _\ \/ __/ _ `/ / _ `/
 *            \____/ /_/ /___/       /___/\__/\_,_/_/\_,_/ 
 * 
 * This file is part of an implementation of the Universe Type System for
 * Scala.
 * 
 * Copyright (C) 2007-2008  Swiss Federal Institute of Technology, Zurich
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 * 
 * 
 * $Id: UniverseTypeSystemPlugin.scala 883 2008-02-01 18:59:56Z ms $
 */
package ch.ethz.inf.sct.uts.plugin

import scala.tools.nsc.Global
import scala.tools.nsc.plugins.Plugin
import ch.ethz.inf.sct.uts.plugin.common._

/**
 * Abstract base plugin for the Scala compiler, containing information about
 * a plugin loaded from a jar file.
 * 
 * @author  Manfred Stock
 * @version $Revision: 883 $
 */
abstract class UniverseTypeSystemPlugin(val global: Global) extends Plugin {
  /**
   * The Universe type system checker component.
   */
  val utscc : UniverseTypeSystemComponentBase

  /**
   * Handle options which are common for all plugins.
   * @param options List of the given options for this plugin.
   * @return the unprocessed options.
   */
  def processCommonOptions(options: List[String]) : List[String] = {
    // Key for the option which is followed by the name of the class which 
    // will contains the defaults
    val defaultsKey = "defaults="
    var unprocessed : List[String] = Nil

    options foreach {
      case "browser"         => utscc.setDisplayBrowser(true)
      case "nobrowser"       => utscc.setDisplayBrowser(false)
      case "loglevel=debug"  => utscc.logger.setLoglevel(utscc.logger.Level.DEBUG)
      case "loglevel=info"   => utscc.logger.setLoglevel(utscc.logger.Level.INFO)
      case "loglevel=notice" => utscc.logger.setLoglevel(utscc.logger.Level.NOTICE)
      case "loglevel=warn"   => utscc.logger.setLoglevel(utscc.logger.Level.WARN)
      case "loglevel=error"  => utscc.logger.setLoglevel(utscc.logger.Level.ERROR)
      case "ast"             => utscc.setPrintAST(true)
      case "noast"           => utscc.setPrintAST(false)
      case o if ((o matches (defaultsKey+".*")) && (o.length > defaultsKey.length)) => 
        UTSDefaults.setDefaults(o substring ((o indexOf defaultsKey) + defaultsKey.length))
      case o                 => unprocessed = o :: unprocessed 
    }
    unprocessed
  }
  
  /** 
   * A description of this plugin's options, suitable as a response
   * to the -help command-line option.
   */
  override val optionsHelp: Option[String] = Some(
      "-P:"+name+":[no]browser           Display Swing-Browser after compiling the code.\n"+
      "-P:"+name+":[no]ast               Print AST after the execution of the plugin.\n"+
      "-P:"+name+":loglevel=<level>      Log at given level and above. Level is one of debug, info, notice, warn, error.\n"+
      "-P:"+name+":defaults=<class>      Class which implements UTSDefaults.\n"
  )
}