/*
 *              __  ____________        ____         __    
 *             / / / /_  __/ __/ ____  / __/______ _/ /__ _
 *            / /_/ / / / _\ \  /___/ _\ \/ __/ _ `/ / _ `/
 *            \____/ /_/ /___/       /___/\__/\_,_/_/\_,_/ 
 * 
 * This file is part of an implementation of the Universe Type System for
 * Scala.
 * 
 * Copyright (C) 2007-2008  Swiss Federal Institute of Technology, Zurich
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 * 
 * 
 * $Id: StaticComponent.scala 883 2008-02-01 18:59:56Z ms $
 */
package ch.ethz.inf.sct.uts.plugin.staticcheck

import scala.tools.nsc._
import scala.tools.nsc.transform._
import scala.collection.mutable.HashMap
import ch.ethz.inf.sct.uts.plugin._
import ch.ethz.inf.sct.uts.plugin.common._
import ch.ethz.inf.sct.uts.annotation._
import ch.ethz.inf.sct.uts.plugin.staticcheck.rules.default._

/**
 * Main part of the Universe type system checker. It traverses the
 * AST and initiates the checks of the type rules.
 * 
 * @author  Manfred Stock
 * @version $Revision: 883 $
 */
class StaticComponent (glbl: Global) extends UniverseTypeSystemComponentBase(glbl) with SymbolStates {
  import global._

  /**
   * Object for access to the Universe type system representation classes, type rules
   * and helper functions.
   */
  var typerules : TypeRules[global.type] = null
  
  /**
   * Options which probably are meant for the type rules.
   */
  private var typerulesOptions : List[String] = Nil
  
  /**
   * Class which contains the implementation of the type rules.
   */
  private var typerulesImplementation = UTSDefaults.typerulesImplementation
  
  /**
   * Set the typerules implementation.
   * @param impl The name of the implementation class.
   */
  def setTyperulesImplementation(impl: String) {
   	typerulesImplementation = impl  
  }
  
  /**
   * Process options which were not accepted by the plugin itself and 
   * may therefore be meant for the used typerules.
   * @param o The option.
   * @return if the option was accepted.
   */
  def processOption(o: String) = {
    typerulesOptions = o :: typerulesOptions
    true
  }
    
  /**
   * Get help about the options implemented by the given typerules.
   * @param name Name of the plugin which got the option.
   * @return the string with help on the options.
   */
  def getOptionsHelp(name: String) : String = {
    if (typerules != null) {
      typerules.getOptionsHelp(name)
    }
    else {
      ""
    }
  }
  
  /**
   * Name of this compiler phase.
   */
  val phaseName = "uts-static"
  
  /**
   * When to execute this phase. Currently done after reference checks.
   */
  val runsAfter = "superaccessors"
  
  /**
   * Factory to create the new phase for the Universe type system checks.
   */
  def newPhase(prev: Phase) = new UTSPhase(prev)
     
  /**
   * Phase which checks if a program fullfills the rules of the Universe type system.
   * @param prev The previous phase.
   */
  class UTSPhase(prev: Phase) extends Phase(prev) {
    /**
     * Name of the phase.
     */
    val name = phaseName
    
    /**
     * Run the Universe type system checking phase.
     */
    def run {
      try {
        typerules = Class.forName(typerulesImplementation)
                         .getConstructor(Array(classOf[Global],classOf[UTSLogger]))
                         .newInstance(Array(global,logger))
                         .asInstanceOf[TypeRules[global.type]]
      }
      catch {
        case e : Exception => 
          logger.notice("WARNING: the specified Universe type system rules implementation class (" +typerulesImplementation +") could not be instantiated, using the default instead.")
          typerules = new DefaultTypeRules[global.type](global,logger)
      }
      typerulesOptions foreach { o => 
        if (! typerules.processOption(o)) {
          logger.notice("Unknown option "+o+" for "+phaseName+".")
        }
      }
      processUnits(
          ("Running Universe type system checks"+(
              if (typerules.getActiveOptions.length > 0) { 
                " using the following options:"+typerules.getActiveOptions.mkString("\n\t- ","\n\t- ","")
              } 
              else {
                "..."
              }
          )),
          (new StaticTypeChecker(typerules)).check
      )
    }
  }
  
  /**
   * Class which does the actual inference and typechecks.
   * @param typerules <code>TypeRules</code> instance which implements the type rules.
   */
  class StaticTypeChecker(val typerules: TypeRules[global.type]) {
    import typerules.{UType,NType,TVarID,ErroneousType,InvalidType,InvalidTypes,RichTreeUTypeList,treeutlist2rutlist,RichUTypeList,utlist2rutlist,ExtendedOwnershipModifier,om2eom,extendedType}
    
    /**
     * Do the actual setup and typecheck.
     */
    def check {
      currentRun.units foreach initSymbolStates
      currentRun.units foreach checkUnit
    }
    
    /**
     * Object with a <code>Traverser</code> instance, used to create new traversers.
     */
    object traverse {
      /**
       * Factory method to create new <code>Traverser</code>s.
       * @param unit The unit which will be processed by the <code>Traverser</code>.
       * @return the new <code>Traverser</code>.
       */
      def newTraverser(unit: CompilationUnit) : Traverser = {
        logger.setUnit(unit)
        new TypingAndInferingTraverser(unit)
      }
    }
    
    /**
     * Check a compilation unit for errors.
     * @unit The unit which is to be checked.
     */
    def checkUnit(unit: CompilationUnit) {
      val traverser = traverse.newTraverser(unit)
      traverser.traverse(unit.body)
    }
      
    /**
     * Find all <code>ValDef</code> and <code>DefDef</code> nodes and store 
     * them in the map of the symbol states. Update the main modifier of 
     * <code>object</code>s.
     * @param unit The unit to process.
     */
    def initSymbolStates(unit: CompilationUnit) {
      val traverser = new ForeachTreeTraverser(tree => {
        tree match {
          case ValDef(_, _, _, _) | DefDef(_, _, _, _, _, _) if tree.symbol.owner.isClass =>
            // Store the symbols of value and method definitions in the hashmap, 
            // together with state information.
            SymbolState.add(tree.symbol -> new SymbolState(tree,unit))
          case ModuleDef(_,_,_) =>
            // Add default modifier to the symbol of 'object'
            if (tree.symbol.enclClass != NoSymbol) {
              addOwnershipModifiers(tree.symbol, UType(tree.symbol.tpe) match {
                case nt: NType =>
                  if (tree.symbol.enclClass.isPackageClass) {
                    // Object is directly inside a package
                    nt.setOwnershipModifier(UTSDefaults.defaultPackageObjectOwnershipModifier)
                  }
                  else {
                    // Object is defined in a class
                    nt.setOwnershipModifier(UTSDefaults.defaultInternalObjectOwnershipModifier)
                  }
                case t         => t
              })
            }
          case _ => 
            ()
        }
      })
      traverser.traverse(unit.body)
    }
  
    /**
     * Phase which checks if a program fullfills the rules of the Universe type system.
     * @param unit Compilation unit this traverser belongs to.
     */
    class TypingAndInferingTraverser(unit: CompilationUnit) extends Traverser  {
      import extendedType._
      import SymbolState._
      
      /**
       * Check if an assignment is ok.
       * @param lhs The <code>Tree</code> which some value is assigned to.
       * @param rhs The <code>Tree</code> which will return the value to be assigned.
       * @return the type of lhs.
       */
      def checkAssignment(lhs: Tree, rhs: Tree) : UType = {
        val lhsRefOwnerType = 
          lhs match {
            case Select(qualifier, selector) => 
              qualifier.asUType            
             case _ => 
              UType(NoType)
           }
  
         val vtype = lhs.asUType
  
         // Check that the value which should be assigned is not an EmptyTree - this happens 
         // when the value is _. If this was ok for Scala, it's ok for us, too.
         if (rhs != EmptyTree) {
           val atype = rhs.asUType
           typerules.checkAssignment(vtype,lhsRefOwnerType,atype)
         }
         else {
           vtype
         }
      }
  
      /**
       * Check the invocation of a method (GT-Invk).
       * @param e0 Receiver of the method call.
       * @param sym Symbol of the called method.
       * @param args Arguments to the method call.
       * @param targs Type arguments to the call which should be checked.
       * @return the return type of the method call, if any.
       */
      def checkInvocation(e0: Tree, sym: Symbol, args: List[Tree], targs: List[Tree]) : UType = {
        inferType(sym)
        val N0 : UType = e0 match {
          // Nested methods need special treatment - they don't really have a target 
          // object they're called on, so set the type to their enclosing class' type
          // (makes some sense, does not really matter though), and their ownership modifier 
          // to 'this' as they are executed in this context.
          case Ident(name) if sym.owner.isMethod => 
            UType(sym.enclClass.tpe) match {
              case nt: NType => nt.setOwnershipModifier(thiz())
              case t         => t 
            }
          case _ => 
            e0.asUType
        }
        val e2 = args map {tree => tree.asUType}
        val ta = targs map {tree => tree.asUType}
        val invalidTypes = N0 :: e2.getInvalid ::: ta.getInvalid
  
        if (logIfInvalid(invalidTypes)) {
          InvalidType(invalidTypes.mkString("","\n",""))
        }
        else {
          typerules.checkInvocation(N0, sym, e2, ta)
        }
      }
      
      /**
       * Check if a given <code>UType</code> is an instance of InvalidType, log an error if this is the case.
       * @param tp The the to check.
       * @return if the type was invalid.
       */
      def logIfInvalid(tp: UType*) : Boolean = {
        logIfInvalid(tp.toList)
      }
      
      /**
       * Check if a given list of <code>UType</code> contains an instance of InvalidType, log
       * errors if this is the case.
       * @param tp The the to check.
       * @return if there was an invalid type.
       */
      def logIfInvalid(tps: List[UType]) : Boolean = {
        tps.foldRight(false)((a,b) => a match {
          case it : ErroneousType => it.log; true
          case _                  =>         b
        })
      }
      
      /**
       * Infer the type of a symbol if it is not known yet.
       * @param symbol The symbol whose type should be inferred.
       */
      def inferType(symbol: Symbol) {
        isTyped(symbol) match {
          case (false,state) if state != null =>         	
            if (!state.isLocked) {
              // Process the sub-AST with the definition of the symbol
              val traverser = StaticTypeChecker.this.traverse.newTraverser(state.unit)
              traverser.traverse(state.code)
              // Reset logger to unit used before processing the definition's subtree
              logger.setUnit(unit)
            }
            else {
              // If we are in a 'recursive' definition, check if the symbol already was annotated - if 
              // yes, accept this type, else this is an error, since we are in a cycle.
              val toInferOn = (if (symbol.tpe.extractType.isTypeVariable) symbol.tpe.extractType.ubgamma else symbol.tpe) match {
                case MethodType(_,resultType) => resultType
                case PolyType(_,resultType)   => resultType
                case t                        => t
              }
              if (! toInferOn.isOwnershipAnnotated && ! toInferOn.isUnit) {
                logger.error("Cycle detected while inferring type of "+symbol+"."+
                  " Presumably involves a recursive definition, so ownership modifier must be given.")
              }              
            }
          case _ =>
        }
      }
      
      /** 
       * Check if a certain symbols is annotated with UTSUnchecked.
       */
      def uncheckedAnnotated(sym: Symbol) = {
         sym.attributes.exists{_.atp.toLongString matches UTSUnchecked().getClass.getName}
      }
      
      /**
       * Infer and check Universe types of a tree.
   		 * @param tree The tree to check.
       * @return the checked and modified tree.
       */
      override def traverse(tree: Tree) = {
        tree match {
          case ClassDef(_,_,_,_) | ModuleDef(_,_,_) => 
            logger.info("------------------------------------------------------------------")
            logger.info("Processing "+tree.symbol)
            logger.info("------------------------------------------------------------------")
          case ValDef(_,_,_,_) | DefDef(_,_,_,_,_,_) if (isTyped(tree.symbol) match { case (typed,state) => !typed && state != null && !state.isLocked }) =>
            lock(tree.symbol)
          case _ => ()
        }
        
        tree match {
          case ValDef(_,_,_,_) | DefDef(_,_,_,_,_,_) if (isTyped(tree.symbol) match { case (typed,state) => typed }) => ()
          case ClassDef(_,name,_,_) if uncheckedAnnotated(tree.symbol)                                                  =>
            logger.notice("Class "+name+" was not checked to fulfill universe type system rules as it was annotated with UTSUnchecked.")
          case _                                                                                                     => {
            // Depth first traversal
            super.traverse(tree)
            logger.enterTree(tree)
  
            try {
              // The pattern match for all the interesting node types of the AST.
              tree match {
                case EmptyTree => 
                case PackageDef(name, stats) =>
                  // package name { stats }
                  logger.debug("Processed package "+name)
                case ModuleDef(mods, name, impl) => 
                  // mods object name impl  where impl = extends parents { defs }; eliminated by refcheck, therefore still exists here
                  logger.debug("Processed static object "+name)
                case ClassDef(mods, name, tparams, impl) =>
                  // mods class name[tparams] impl
                  // Do not check universe type system rules if the class was annotated with UTSUnchecked
                  logIfInvalid(typerules.checkClassDefinition(tree.symbol, tparams map {t => (t.pos,UType(t.symbol.tpe))}))
                case ValDef(mods, name, tpt, rhs) =>
                  // mods val name: tpt = rhs
                  logger.debug("Valdef: "+name+": "+UType(tree.symbol.tpe)+" = "+rhs+"; TreeSymbol: "+tree.symbol)
                  logIfInvalid(tpt.asUType)
                  val rhstpe = rhs.asUType
                  // Add ownership modifiers which might be missing on the left hand side
                  addOwnershipModifiers(tree.symbol,rhstpe)
                  // Check the value definition
                  logIfInvalid(typerules.checkValueDefinition(tree.symbol,rhstpe))
  
                  // If the owner if the symbol is a setter method, ie. this is 
                  // the parameter of a setter method, its ownership modifiers need to
                  // be updated since they need to reflect those of the symbol the setter 
                  // belongs to.
                  if (tree.symbol.owner.isSetter) {
                    val accessed = tree.symbol.owner.accessed
                    addOwnershipModifiers(tree.symbol, UType(accessed.tpe))
                  }
                  
                  // Type of symbol known by now
  								setTyped(tree.symbol)
                  unlock(tree.symbol)
                case DefDef(mods, name, tparams, vparamss, tpt, rhs) =>
                  // mods def name[tparams](vparams): tpt = rhs
                  logger.debug("Function/Method definition: "+name)
                  
                  // If method is a setter, parameter type might need to be updated 
                  // to reflect the annotation of the accessed field.
                  if (tree.symbol.isSetter) {
                    val accessed = tree.symbol.accessed
                    tree.symbol.tpe match {
                      case MethodType(paramTypes, resultType) =>
                      	tree.symbol.updateInfo(MethodType(List(addOwnershipModifiers(paramTypes(0),UType(accessed.tpe))),resultType))
                      case _ => 
                    }
                  }
                  
                  val typeparams  = (tparams map {t => UType(t.symbol.tpe)})
                  val valueparams = vparamss map {
                    _ map {
                      t => UType(t.symbol.tpe)
                    }
                  }
                  
                  // Get formal return type
                  var frtype = tpt.asUType
                  
                  // Actual return type - set it to the formal return type if there is 
                  // no implementation or the method is abstract
                  val artype = if (rhs == EmptyTree || tree.symbol.isIncompleteIn(tree.symbol.owner)) {
                                 frtype
                               } 
                               else {
                                 val t = rhs.asUType
                                 addOwnershipModifiers(tree.symbol, t)
                                 frtype = t.addOwnershipModifiersOn(frtype)
                                 t
                               }
  
                  logIfInvalid(typerules.checkMethodDefinition(tree.symbol,name.toString,typeparams,valueparams,frtype,artype))
                  
                  // Type of symbol known by now
                  setTyped(tree.symbol)
                  unlock(tree.symbol)
                case TypeDef(mods, name, tparams, rhs) => 
                  // mods type name[tparams] = rhs
                  logger.debug("Type definition: "+name +" => "+tree.symbol.tpe)
                  logIfInvalid(
                      rhs.asUType :: UType(tree.symbol.tpe) :: (tparams map {p => p.asUType})
                  )
                case Block(stats, expr) =>
                  // { stats; expr }  
                  tree.addOwnershipModifier(expr.asUType)
                case Function(vparams, body) =>
                  // vparams => body  where vparams:List[ValDef]
                  // Anonymous function: vparams => body  where vparams:List[ValDef]
                  logIfInvalid(vparams map {p => UType(p.symbol.tpe)})
                  logIfInvalid(body.asUType)
                  val annotatedType = tree.tpe match {
                    case TypeRef(pre,sym,args) => typeRef(pre,sym,{
                                                    val rev = args.reverse
                                                    val hd = rev.head
                                                    val tl = rev.tail
                                                    (addOwnershipModifiers(hd,body.asUType) :: tl).reverse
                                                  }) 
                  }
                  tree.updateType(annotatedType)
                case Assign(lhs, rhs) =>
                  // lhs = rhs
                  logIfInvalid(checkAssignment(lhs,rhs))
                case If(cond, thenp, elsep) =>
                  // if (cond) thenp else elsep
                  val iftpe   = tree.asUType 
                  val condtpe = cond.asUType
                  val thentpe = thenp.asUType
                  val elsetpe = if (elsep != EmptyTree) { elsep.asUType } else UType(NoType)
                  if (! logIfInvalid(iftpe,condtpe,thentpe,elsetpe)) {
                    tree.addOwnershipModifier(typerules.checkIfThenElse(iftpe,condtpe,thentpe,thenp.pos,elsetpe,elsep.pos))
                  }
                case Match(selector, cases) =>
                  // selector match { cases }
                  // Create a mapping from case-blocks to checked types
                  val ctypes = cases map {c => (c,c.asUType)}
                  if (ctypes.getInvalid.isEmpty) {
                    val mtype = tree.asUType
                    logIfInvalid(mtype)
                    tree.addOwnershipModifier(typerules.checkMatch(mtype,ctypes map {case (c,t) => (c.pos,t)}))
                  }
                  else {
                    InvalidTypes(List.flatten((ctypes getInvalid) map {
                      case (c,t: InvalidType)  => List(t.setPosition(c.pos))
                      case (c,t: InvalidTypes) => t.errors
                    })).log
                  }
                case Bind(name,body) =>
                  // name @ pat
                  logIfInvalid(body.asUType)
                case CaseDef(pat, guard, body) =>
                  // case pat if guard => body
                  val bodytype = body.asUType
                  logIfInvalid(pat.asUType, guard.asUType)
                  tree.addOwnershipModifier(bodytype)
                case Return(expr) =>
                	// return expr
                  tree.addOwnershipModifier(expr.asUType)
                case Try(block, catches, finalizer) =>
                  // try block catch { catches } finally finalizer where catches: List[CaseDef]
                  val blockres = block.asUType
                  val finalizertype = if (finalizer != EmptyTree) { finalizer.asUType } else null
                  val trytype = tree.asUType
                  val catchestypes = catches map { c => (c.pos,c.asUType)}
                  typerules.checkTry(trytype, blockres, block.pos, catchestypes, finalizertype) match {
                    case et: ErroneousType => et.log
                    case t                 => tree.addOwnershipModifier(t)  
                  }
                case Throw(expr) =>
                  // throw expr
                  tree.addOwnershipModifier(expr.asUType)  
                case New(tpt) =>
                  // new tpt, always in the context: new tpt.<init>[targs](args)
                  logger.debug("New: "+tpt.tpe)
                  tpt.asUType match {
                    case et: ErroneousType => 
                      et.log
                    case nt: NType =>
                      logIfInvalid(typerules.checkNew(nt))                  
                    case tv: TVarID =>
                      logger.error("Cannot create new instance of a type variable.")
                  }
                case Typed(expr, tpt) =>
                  // expr: tpt
                  val exptpe = expr.asUType
                  val tpttpe = tpt.asUType
                  logIfInvalid(exptpe,tpttpe)
                  logIfInvalid(typerules.checkTyped(exptpe, tpttpe))
                case Apply(fun@Select(e0, m), args) =>
                  // fun(args)
                  // Function/method call on given object e0
                  logger.debug("Apply on Select: Treesymbol: "+tree.symbol+"; Function: "+fun+" (type: "+fun.tpe+") -> Select("+e0+" ("+e0.tpe+"),"+m+"); args: "+args)
                  tree.addOwnershipModifier(checkInvocation(e0,tree.symbol,args,Nil))
                case Apply(tfun@TypeApply(fun@Select(e0,m),targs),args) =>
                  // fun[targs](args)
                  // Parameterized function/method call on given object
                  logger.debug("Apply with TypeApply on Select: Treesymbol: "+tree.symbol)
                  tree.addOwnershipModifier(checkInvocation(e0,tree.symbol,args,targs))
                case Apply(fun, args) => 
                  // fun(args)
                  // Function/method call, eg. constructors of case-classes, nested methods
                  logger.debug("Apply: fun: "+fun+" args: "+args+" Tree type: "+tree.tpe+" Tree Symbol: "+tree.symbol+" fun symbol: "+fun.symbol+" tree: "+tree)
                  if (tree.symbol != NoSymbol) {
                    val tp = checkInvocation(fun,tree.symbol,args,Nil)
                    logIfInvalid(tp)
                    if (!tree.tpe.isCaseClass) {
                      // Don't add ownership modifiers if the type of the node is from a case class, ie. 
                      // this Apply was from a constructor of a case class. Annotations don't seem to be allowed 
                      // there (one can't write them explicitly, and if adding them here, it crashes some 
                      // later phase...).
                      tree.addOwnershipModifier(tp)                    
                    }
                  }
                case TypeApply(fun@Select(e0,m),targs) =>
                  // fun[targs]
                  logger.debug("Type arguments: "+targs+" to call of "+fun)
                  tree.addOwnershipModifier(checkInvocation(e0,tree.symbol,Nil,targs))
                case Super(qual, mix) =>
                  // qual.super[mix]
                  val symtype = UType(tree.symbol.tpe)
                  logIfInvalid(symtype)
                  tree.addOwnershipModifier(symtype)
                case This(qual) =>
                  // qual.this
                  logger.debug("This: "+tree+" tpe: "+tree.tpe)
                  tree.addOwnershipModifier(typerules.checkThis(UType(tree.symbol.tpe)))
                case Select(qualifier, selector) =>
                  // qualifier.selector
                	logger.debug("Processing "+qualifier+"."+selector+"...")
                  if (tree.symbol.isMethod) {
                    // Simple field accesses usually are method calls, esp. if the field's symbol is a method.
                    tree.addOwnershipModifier(checkInvocation(qualifier,tree.symbol,Nil,Nil))
                  }
                  else {
                    inferType(tree.symbol)
                    tree.addOwnershipModifier(typerules.checkSelect(qualifier.asUType,tree.symbol))
                  }
                case Ident(name) =>
                  logger.debug("Ident: "+name+{ if (tree.symbol.tpe != NoType) { ": "+UType(tree.symbol.tpe) } else { "" }})
                  // tree.symbol.tpe is NoType in the catch-block of a try-statement, the required type then is in tree1.tpe
                  tree.addOwnershipModifier(
                      if (tree.symbol.tpe != NoType) {
                        inferType(tree.symbol)
                        UType(tree.symbol.tpe)
                      }
                      else {
                        tree.asUType
                      }
                  )
                case _ =>
              } 
            }
            finally {
              logger.leaveTree
            }
          }
        }
      }
    }
        
    /**
     * Class to add some useful methods to tree nodes.
     * @param tree The tree node which should be wrapped.
     */
    class RichTree(tree: Tree) {
      /**
       * Get the type of a tree node.
       * @param tree Tree node whose type is requested.
       * @return the <code>UType</code> representation of the tree's type.
       */
      def asUType : UType = UType(tree.tpe)
      
      /**
       * Update the type of the contained tree.
       * @param tpe The type which should be set for the tree.
       */
      def updateType(tpe: Type) = tree.setType(tpe)
      
      /**
       * Add an ownership modifier annotation to the type of a tree. This is only done if the 
       * type of the tree was not annotated. Log an error if the source type is an 
       * <code>ErroneousType</code>.
       * @param tp The type which contains the annotation.
       * @return the modified tree.
       */
      def addOwnershipModifier(tp: UType) : Tree = {
        tp match {
          case nt : NType         => tree.updateType(addOwnershipModifiers(tree.tpe, tp))
          case it : ErroneousType => it.log
          case _                  =>
        }
        tree
      }
    }
    
    /**
     * Implicitly convert tree nodes to <code>RichTree</code>s.
     * @param tree The tree node which should be enriched. 
     */
    implicit def enrichTree(tree: Tree) = new RichTree(tree)
    
    /**
     * Add an ownership modifier annotation to the type of a symbol. This is only done if the 
     * symbols type was not annotated. Log an error if the source type is an 
     * <code>ErroneousType</code>.
     * @param sym The <code>Symbol</code> which should get the annotation.
     * @param tpe The type which contains an annotation.
     */
    def addOwnershipModifiers(sym: Symbol, tp: UType) {
      tp match {
        case nt : NType         => sym.updateInfo(addOwnershipModifiers(sym.tpe, nt.setOwnershipModifier(nt.om.parentMainmodifier))) 
        case it : ErroneousType => it.log
        case _                  => 
      }
    }
            
    /**
     * Add an ownership modifier annotation to a <code>Type</code>. This is only done if the 
     * compiler type was not annotated. Log an error if the source type is an 
     * <code>ErroneousType</code>.
     * @param t The <code>Type</code> which should get the annotation.
     * @param ut The type which contains an annotation.
     * @return the modified <code>Type</code>
     */
    def addOwnershipModifiers(t: Type, ut: UType) : Type = {
      ut match {
        case nt : NType         =>
          logger.debug("Adding ownership modifier "+nt.om+" to type "+t)
          val res = nt.addOwnershipModifiersOn(t)
          logger.debug("Added modifiers from "+nt+": "+t+" ("+t.getClass+") => "+res)
          res
        case it : ErroneousType => 
          it.log
          t
        case _                  =>
          t
      }
    }
  }
}