/*
 *              __  ____________        ____         __    
 *             / / / /_  __/ __/ ____  / __/______ _/ /__ _
 *            / /_/ / / / _\ \  /___/ _\ \/ __/ _ `/ / _ `/
 *            \____/ /_/ /___/       /___/\__/\_,_/_/\_,_/ 
 * 
 * This file is part of an implementation of the Universe Type System for
 * Scala.
 * 
 * Copyright (C) 2007-2008  Swiss Federal Institute of Technology, Zurich
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 * 
 * 
 * $Id: ImplementationBase.scala 817 2008-01-25 19:44:05Z ms $
 */
package ch.ethz.inf.sct.uts.rt

import ch.ethz.inf.sct.uts.rt.implementation.Modifiers

/**
 * Implementation base for the runtime classes.
 * 
 * @author  Manfred Stock
 * @version $Revision: 817 $
 */
trait ImplementationBase {
  /**
   * Set the owner during execution of the constructor.
   * @param obj The object whose owner should be set.
   * @return the newly created entry.
   */
  def setOwner(obj: AnyRef)
       
  /**
   * Set the owner of object <code>obj</code> to object <code>owner</code>.
   * @param obj The object whose owner should be set.
   * @param owner The object which will become the owner of <code>obj</code>.
   */
  def setOwnerRep(obj: AnyRef, owner: AnyRef)
      
  /**
   * Set the owner of object <code>obj</code> to the same owner as 
   * the owner of object <code>peer</code>.
   * @param obj Object whose owner should be set.
   * @param peer Object which will be a peer ob <code>obj</code>.
   */
  def setOwnerPeer(obj: AnyRef, peer: AnyRef)

  /**
   * Set some data which will be used during the constructor execution for 
   * setting the owner of objects created during constructor execution.
   * @param currentObject Object which contained the call to <code>new</code>.
   * @param cls Class which should be instantiated.
   * @param mod Main ownership modifier in the <code>new</code> call. 
   */
  protected def setConstructorData[T](currentObject: AnyRef, cls: java.lang.Class[T], mod: Modifiers.Value)
  
  /**
   * Set some data which will be used during the constructor execution for 
   * setting the owner of objects created during constructor execution. Used 
   * if the main modifier was <code>@peer</code>.
   * @param currentObject Object which contained the call to <code>new</code>.
   * @param cls Class which should be instantiated.
   */
  def setConstructorDataPeer[T](currentObject: AnyRef, cls: java.lang.Class[T]) {
    setConstructorData(currentObject, cls, Modifiers.PEER)
  }
   
  /**
   * Set some data which will be used during the constructor execution for 
   * setting the owner of objects created during constructor execution. Used 
   * if the main modifier was <code>@rep</code>.
   * @param currentObject Object which contained the call to <code>new</code>.
   * @param cls Class which should be instantiated.
   */
  def setConstructorDataRep[T](currentObject: AnyRef, cls: java.lang.Class[T]) {
    setConstructorData(currentObject, cls, Modifiers.REP)
  }
  
  /**
   * Check if a given object is the owner of another object.
   * @param in Object which could be the owner of <code>obj</code>.
   * @param obj Object which could be owned by <code>in</code>.
   * @return if <code>in</code> is the owner of <code>obj</code>.
   */
  def isOwner(in: Any, obj: Any) : Boolean
      
  /**
   * Check if a given object is a peer of another object.
   * @param obj0 Object which could be a peer of <code>obj1</code>.
   * @param obj1 Object which could be a peer of <code>obj0</code>.
   * @return if <code>obj0</code> is a peer of <code>obj1</code>.
   */
  def isPeer(obj0: Any, obj1: Any) : Boolean
   
  /**
   * Check if a given object is the owner of another object. Throw exception 
   * if this is not the case. 
   * @param in Object which should be the owner of <code>obj</code>.
   * @param obj Object which should be owned by <code>in</code>.
   */
  def assertRepCastable(in: Any, obj: Any) {
  	if (!isOwner(in, obj))
      UTSRuntime.policy.illegalRepCast
  }
   
  /**
   * Check if a given object is a peer of another object. Throw exception 
   * if this is not the case. 
   * @param obj0 Object which should be a peer of <code>obj1</code>.
   * @param obj1 Object which should be a peer of <code>obj0</code>.
   */
  def assertPeerCastable(obj0: Any, obj1: Any) {
    if (!isPeer(obj0, obj1))
      UTSRuntime.policy.illegalPeerCast
  }
 
  /**
   * Get some information about the runtime, eg. the mapping from objects to their owners.
   * @return some information.
   */
  def info : String
}
