/*
 *              __  ____________        ____         __    
 *             / / / /_  __/ __/ ____  / __/______ _/ /__ _
 *            / /_/ / / / _\ \  /___/ _\ \/ __/ _ `/ / _ `/
 *            \____/ /_/ /___/       /___/\__/\_,_/_/\_,_/ 
 * 
 * This file is part of an implementation of the Universe Type System for
 * Scala.
 * 
 * Copyright (C) 2007-2008  Swiss Federal Institute of Technology, Zurich
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 * 
 * 
 * $Id: DefaultImplementation.scala 817 2008-01-25 19:44:05Z ms $
 */
package ch.ethz.inf.sct.uts.rt.implementation

import java.lang.ref.ReferenceQueue

/**
 * Default implementation for the Universe type system runtime checks.
 * 
 * @author  Manfred Stock
 * @version $Revision: 817 $
 */
class DefaultImplementation extends ImplementationBase {
  /**
   * The <code>ReferenceQueue</code> used for the cleanup of deleted objects.
   */
  val queue = new ReferenceQueue[AnyRef]

  /**
   * Hashtable which stores the ownership relations.
   */
  private val table = new Hashtable
  
  /**
   * Hashtable entry for the <code>UTSRuntime</code> which is used as a dummy owner 
   * if no other owner is available.
   */
  private val rtEntry = table.put(this,new HashtableEntry(new UrtWeakReference(this)))
     
  // Add current Thread and ReferenceQueue to the table
  setOwner(Thread.currentThread,rtEntry)
  setOwner(queue,rtEntry)

  /**
   * Set the owner of object <code>obj</code> to object <code>owner</code>.
   * @param obj Object whose owner should be set.
   * @param owner Object which will become the owner of <code>obj</code>.
   */
  private def setOwner(obj: AnyRef, owner: HashtableEntry) : HashtableEntry = {
    table.get(obj) match {
      case Some(entry) => entry
      case None        => 
        val res = table.put(obj,owner)
        owner.children += 1
        /* 
   			 * Reset the threads saved data, so constructors or functions called
   			 * from non-universe type system aware code are handled correctly.
   			 */
        val thread = getCurrentThreadEntry
        thread.currentObject = owner.obj.get
        thread.objectClass = classOf[AnyRef]
        thread.modifier = Modifiers.PEER
        
        /*
         *  Check for collected objects and remove unnecessary
         *  stuff in our data structure to make it ready to be
         *  collected too.
         */
        var o : UrtWeakReference = queue.poll.asInstanceOf[UrtWeakReference]        
        while (o != null) {
          var i = table.get(o) match {
            case Some(entry: HashtableEntry) => entry
            case None                        => null
          }
          /* clear WeakLink so the Object can be collected */
          i.obj = null
          /* the second test is for further iterations */
          while ((i != null) && (i.children == 0) && (i.obj == null) ) {
            table.remove(i)
            i.owner.children -= 1

            /*
             * If the owner has no more children now, and
             * its object already has been collected, we
             * may delete this one too.
             */
            i = i.owner
          }
          o = queue.poll.asInstanceOf[UrtWeakReference]
        }
        res
    }
  }
   
  /**
   * Set the owner during execution of the constructor. Get owner and class 
   * from the current thread.
   * @param obj The object whose owner should be set.
   * @return the newly created entry.
   */
  def setOwner(obj: AnyRef) {
    val thread = getCurrentThreadEntry
    val cls = obj match {
      case ar: AnyRef => ar.getClass
    }
    if (cls == thread.objectClass) {
      thread.modifier match {
        case Modifiers.REP  => setOwnerRep(obj,thread.currentObject)
        case Modifiers.PEER => setOwnerPeer(obj,thread.currentObject)
      }
    }
    else {
      val owner = UTSRuntime.policy.getNativeOwner(obj)
      if (owner == null) {
        setOwnerPeer(obj, thread.currentObject)
      }
      else {
        setOwnerRep(obj, owner);
      }
    }
  }
    
  /**
   * Set the owner of object <code>obj</code> to object <code>owner</code>.
   * @param obj The object whose owner should be set.
   * @param owner The object which will become the owner of <code>obj</code>.
   */
  def setOwnerRep(obj: AnyRef, owner: AnyRef) = {
    table.get(owner) match {
      case Some(ownerobj) => setOwner(obj,ownerobj)
      case None           => throw new UTSRuntimeException("Owner of the object has not yet been registered.") 
    }
  }
   
  /**
   * Set the owner of object <code>obj</code> to the same owner as 
   * the owner of object <code>peer</code>.
   * @param obj Object whose owner should be set.
   * @param peer Object which will be a peer ob <code>obj</code>.
   */
  def setOwnerPeer(obj: AnyRef, peer: AnyRef)  = {
    table.get(peer) match {
      case Some(peerobj) => setOwner(obj,peerobj.owner)
      case None          => throw new UTSRuntimeException("Peer "+peer+" of the object "+obj+" has not yet been registered.")
    }
  }
  
  /**
   * Set some data which will be used during the constructor execution for 
   * setting the owner of objects created during constructor execution.
   * @param currentObject Object which contained the call to <code>new</code>.
   * @param cls Class which should be instantiated.
   * @param mod Main ownership modifier in the <code>new</code> call. 
   */
  protected def setConstructorData[T](currentObject: AnyRef, cls: java.lang.Class[T], mod: Modifiers.Value) {
    val cobj = table.get(currentObject) match {
      case Some(obj) => obj
      case None      => 
        val owner = {
          val nativeOwner = UTSRuntime.policy.getNativeOwner(currentObject) 
          if (nativeOwner == null) { 
            rtEntry
          } 
          else { 
            new HashtableEntry(new UrtWeakReference(nativeOwner)) 
          }  
        }
        setOwner(currentObject, owner)
    }
    val thread = getCurrentThreadEntry
    thread.currentObject = currentObject
    thread.objectClass = cls
    thread.modifier = mod
  }

  /**
   * Get the current Threads HashtableEntry or create it
   * if it does not exist.
   * (It may not exist if the Thread has been created outside of universe
   * type system aware code)
   * 
   * @return the current threads <code>UrtHashtableThreadEntry</code>.
   */
  private def getCurrentThreadEntry : HashtableThreadEntry = {
    table.get(Thread.currentThread) match {
      case Some(thread: HashtableThreadEntry) => thread
      case None                               => setOwner(Thread.currentThread, rtEntry).asInstanceOf[HashtableThreadEntry]
    }
  }
     
  /**
   * Get string representation of the table.
   * @return the string representation of the table.
   */
  def info : String = {
    table.toString
  }
     
  /**
   * Check if a given object is the owner of another object.
   * @param in Object which could be the owner of <code>obj</code>.
   * @param obj Object which could be owned by <code>in</code>.
   * @return if <code>in</code> is the owner of <code>obj</code>.
   */
  def isOwner(in: Any, obj: Any) : Boolean = {
    /* null objects neither have nor are owners */
    if ((in == null) || (obj == null)) {
      false
    }
    else {
      getOwnerEntry(obj) match {
        case Some(owner0) => table.get(in) match {
          case Some(entry1) => entry1 == owner0
          case None         => false
        }
        case None         => false
      }
    }
  }
   
  /**
   * Check if a given object is a peer of another object.
   * @param obj0 Object which could be a peer of <code>obj1</code>.
   * @param obj1 Object which could be a peer of <code>obj0</code>.
   * @return if <code>obj0</code> is a peer of <code>obj1</code>.
   */
  def isPeer(obj0: Any, obj1: Any) : Boolean = {
    /* null objects neither have nor are owners */
    if ((obj0 == null) || (obj1 == null)) {
      false
    }
    else {
      getOwnerEntry(obj0) match {
        case Some(owner0) => getOwnerEntry(obj1) match {
          case Some(owner1) => owner0 == owner1
          case None         => UTSRuntime.policy.isExternalPeer
        }
        case None         => UTSRuntime.policy.isExternalPeer
      }
    }
  }
   
  /**
   * Get the hashtable entry of an objects owner.
   * @param obj  The object.
   * @return the hashtable entry of the objects owner.
   */
  private def getOwnerEntry(obj: Any) : Option[HashtableEntry] = {
    /* 
     * Every object should be in the table
     * if not, it has been created outside of universe type system aware code
     */
    table.get(obj) match {
      case Some(entry) =>
        /* every object should have an owner */
        assert(entry.owner != null)
        Some(entry.owner)
      case None        => None
    }
  }
}