/*
 *              __  ____________        ____         __    
 *             / / / /_  __/ __/ ____  / __/______ _/ /__ _
 *            / /_/ / / / _\ \  /___/ _\ \/ __/ _ `/ / _ `/
 *            \____/ /_/ /___/       /___/\__/\_,_/_/\_,_/ 
 * 
 * This file is part of an implementation of the Universe Type System for
 * Scala.
 * 
 * Copyright (C) 2007-2008  Swiss Federal Institute of Technology, Zurich
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 * 
 * 
 * $Id: UniverseTypeSystemComponentBase.scala 883 2008-02-01 18:59:56Z ms $
 */
package ch.ethz.inf.sct.uts.plugin

import scala.tools.nsc._
import scala.tools.nsc.transform._
import scala.tools.nsc.plugins.PluginComponent
import ch.ethz.inf.sct.uts.plugin.common._
import ch.ethz.inf.sct.uts.annotation._

/**
 * Abstract base class for the main component of the Universe type system checker.
 * 
 * @author  Manfred Stock
 * @version $Revision: 883 $
 */
abstract class UniverseTypeSystemComponentBase (val global: Global) extends PluginComponent {
  import global._

  /**
   * Instance of a <code>UTSLogger</code> for creating errors, warnings 
   * and generals debug/info messages.
   */
  object logger extends UTSLogger {
    val compiler: global.type = global
  }
  
  /**
   * Flag if the browser should be displayed.
   */
  private var dispBrowser = false
  
  /**
   * Activate display of browser after the compilation.
   * @param b If the browser should be displayed or not.
   */
  def setDisplayBrowser(b: Boolean) {
    dispBrowser = b
  }
  
  /**
   * Show the browser with the AST.
   * @param units The units which should be shown.
   */
  def showASTBrowser {
    if (dispBrowser) {
      global.treeBrowsers.create.browse(currentRun.units) 
    }
  }
  
  /**
   * Should the AST be printed after processing it?
   */
  var printAST = false
  
  /**
   * Set if the AST should be printed after executing the plugin.
   * @param b If the AST should be printed.
   */
  def setPrintAST(b: Boolean) {
    printAST = b
  }
 
  /**
   * Actually print the processed AST.
   */
  def printProcessedAST {
    if (printAST) {
      currentRun.units foreach { u =>
      	val title = "AST of "+u.source.path+":"
        val line = List.make(title.length,"+") mkString ("","","")
        logger.notice(List(line,title,line) mkString ("","\n",""))
        logger.info(u.body.toString)
      }
    }
  }
  
  /**
   * Process the units of the program.
   * @param message Message describing the phase.
   * @param f       Code block which does the actual processing.
   */
  def processUnits(message: => String, f: => Unit) {
    if (!settings.Xplugtypes.value) {
      logger.notice("-Xplug-types not in use, therefore no Universe type system checks are done.")
    }
    else {
      logger.info(message)
      f
      logger.info("Done.")
     
      showASTBrowser
      printProcessedAST
    }
  }
}