/*
 *              __  ____________        ____         __    
 *             / / / /_  __/ __/ ____  / __/______ _/ /__ _
 *            / /_/ / / / _\ \  /___/ _\ \/ __/ _ `/ / _ `/
 *            \____/ /_/ /___/       /___/\__/\_,_/_/\_,_/ 
 * 
 * This file is part of an implementation of the Universe Type System for
 * Scala.
 * 
 * Copyright (C) 2007-2008  Swiss Federal Institute of Technology, Zurich
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 * 
 * 
 * $Id: ExtendedType.scala 883 2008-02-01 18:59:56Z ms $
 */
package ch.ethz.inf.sct.uts.plugin.common

import scala.tools.nsc._

import ch.ethz.inf.sct.uts.annotation._
import ch.ethz.inf.sct.uts.plugin.common._

/**
 * Trait providing an extended representation of the Scala compiler's 
 * <code>Type</code>.
 *
 * @author  Manfred Stock
 * @version $Revision: 883 $
 */
trait ExtendedType {
  import UTSDefaults._
  
  /**
   * Reference to the <code>Global</code> instance the plugin got instantiated with.
   */
  val global: Global
  import global._
  
  /**
   * Mapping from strings to actual ownership modifier instances. Used to convert the 
   * annotations of a type to instances which are used from there on. 
   */
  final private val userOwnershipModifierAnnotations: Map[String,OwnershipModifier] 
    = Map[String,OwnershipModifier](
        (rep().getClass.getName)  -> rep(),
        (peer().getClass.getName) -> peer(),
        (any().getClass.getName)  -> any(),
        (some().getClass.getName) -> some(),
        (thiz().getClass.getName) -> thiz(),
        (lost().getClass.getName) -> lost()
    )
    
  /**
   * Enriched variant of Scala's type representation.
   * @param t The type to enrich which is encapsulated by this class.
   */
  class RichType(val tpe: Type) {
    /**
     * Get instances of the ownership modifiers of this type.
     * @return the ownership modifiers of the type.
     */
    val ownershipModifiers : List[OwnershipModifier] = {
      val ownershipAnnotations = 
        tpe.attributes.map(a => a.atp.toLongString)
                      .filter(a => userOwnershipModifierAnnotations.get(a) match {
                        case Some(om) => true
                        case None     => false
                      })
      ownershipAnnotations.map(userOwnershipModifierAnnotations.get(_) match {
        case Some(om) => om
        case None     => null
      }).filter(_ != null)
    }

    /**
     * Does this type have any ownership annotations?
     */
    val isOwnershipAnnotated = ! ownershipModifiers.isEmpty
    
    /**
     * Checks if this type contains an ownership annotation - either on the 
     * top level, or somwhere in it's type arguments.
     */
    def containsOwnershipAnnotation : Boolean = {
      isOwnershipAnnotated || (tpe.typeArgs exists {t => t.containsOwnershipAnnotation})
    }
    
    /**
     * Check if the type is a type variable.
     */
    val isTypeVariable = tpe.typeSymbol.isTypeParameterOrSkolem
    
    /**
     * Check if a parent of the given type is AnyVal.
     */
    private[ExtendedType] def hasAnyValParent = tpe.parents.exists(_ =:= definitions.AnyValClass.tpe)
    
    /**
     * Check if the type is a value type, ie. it extends scala.AnyVal.
     * @return if this type is a value type.
     */
    def isValueType = tpe.hasAnyValParent || extractType.tpe.hasAnyValParent
    
    /**
     * Check if the type is a value type or an otherwise immutable type such
     * as <code>java.lang.String</code>.
     * @return if the type is immutable.
     */
    def isImmutable = isValueType || ((tpe.typeSymbol != NoSymbol) && immutableTypes.contains(tpe.typeSymbol.fullNameString))
    
    /**
     * Compare two <code>RichType</code> instances. This method is here for the case that 
     * the compiler's type cannot simply be compared the way we need it using =:=.
     * @param that Type to compare the encapsulated one to.
     * @return if the given classes are the same.
     */
    def sameClassAs(that: RichType) : Boolean = {
      this.tpe.typeSymbol.tpe =:= that.tpe.typeSymbol.tpe
    }
     
    /**
     * Get the methods of this type, if any.
     * @return a list of the methods of this class. 
     */
    def methods = {
      var res : List[Symbol] = Nil
      tpe.members.foreach(s => if (s.isMethod) { 
        res = res ::: List(s)
      })
      res
    }
    
    /**
     * Get the name of a type.
     */
    val name = this.tpe.typeSymbol.rawname
    
    /**
     * The <code>Type</code> object of the class of this type - type variables etc. should
     * not yet be replaced in there.
     * @return the type of the class.
     */
    def getClassType : Type = {
      tpe match { 
        case SingleType(pre, sym) =>
          // pre.sym.type
          pre.memberType(sym)
        case TypeRef(pre, sym, args) =>
          // pre.sym[targs]
          pre.memberType(sym)
        case AnnotatedType(attribs, tp, selfsym) =>
          // tp @attribs
          tp.getClassType
        case tp => tp
      }
    }
     
    /**
     * Check if this type is <code>scala.Unit</code>.
     * @return if this type is <code>scala.Unit</code>.
     */
    def isUnit = definitions.UnitClass.tpe =:= tpe
    
    /**
     * Check if this type is <code>scala.Null</code>.
     * @return if this type is <code>scala.Null</code>. 
     */
    def isScalaNull = definitions.AllRefClass.tpe =:= tpe.deconst
     
    /**
     * Check if this type is <code>scala.Nothing</code>, the common subtype of all types.
     * @return if this type is <code>scala.Nothing</code>.
     */
    def isScalaNothing = definitions.AllClass.tpe =:= tpe.deconst
     
    /**
     * If this type is a <code>TypeRef</code>, extract the first argument from it if the symbol in 
     * the <code>TypeRef</code> matches the given symbol, null in the other cases.
     * @param trsym The symbol which should be contained in the <code>TypeRef</code>
     */
    private def unwrapTypeRef(trsym: Symbol) = {
      tpe match {
        case TypeRef(pre,sym,args) => 
          if (sym == trsym && !args.isEmpty) {
            args(0)
          }
          else {
            null
          }
        case _ => null
      } 
    }
      
    /**
     * Extract the actual type if the type is more complex, i.e. a method (return the return type), 
     * a repeated- or byname-method.
     * @return the contained type of the more complex type.
     */
    def extractType : Type = {
      tpe match {
        case mt: MethodType         => mt.resultType
        case tp: Type if isRepeated => repeatedType
        case tp: Type if isByName   => byNameType
        case SingleType(pre,sym)    => sym.tpe.extractType
        case tp                     => tp
      }
    }
      
    /**
     * If the type  is being repeated.
     * @return the type which is being repeated.
     */
    def repeatedType = unwrapTypeRef(definitions.RepeatedParamClass)
    
    /**
     * If this type is a repeated type.
     * @return if this type is a repeated type.
     */
    def isRepeated = repeatedType != null
        
    /**
     * Get the type which is contained by name.
     * @return the type contained by name.
     */
    def byNameType = unwrapTypeRef(definitions.ByNameParamClass)
      
    /**
     * If this type is a byname type.
     * @return if this type is a ByName type.
     */
    def isByName = byNameType != null
    
    /**
     * Return the upper bound of this type.
     * @return the upper bound.
     */
    def gamma : Type = {
      tpe.bounds match {
        case TypeBounds(lo,hi) => hi
      }
    }
        
    /**
     * Find upper type bound which is not a type variable by going upwards using gamma.
     * @return the upper bound of this type.
     */
    def ubgamma : Type = {
      gamma match { 
        case NoType                  => NoType
        case t if (t.isTypeVariable) => t.ubgamma
        case t                       => t
      }
    }
    
    /**
     * Check if the class of this type is a case class (taken from PatternNodes.scala)
     * @return if the class of the given type is a case class.
     */
    def isCaseClass : Boolean =
      tpe match {
        case TypeRef(_, sym, _) =>
          if(!sym.isAliasType)
            sym.hasFlag(symtab.Flags.CASE)
          else
            tpe.normalize.typeSymbol.hasFlag(symtab.Flags.CASE)
        case _ => false
    }
  }

  /**
   * Implicit function to convert a compiler type to the enriched type.
   * @param t Type to enrich.
   * @return the enriched type.
   */
  implicit def enrichType(t: Type) = new RichType(t)
  
  /**
   * Convert an ownership modifier to an <code>AnnotationInfo</code> instance.
   * @param om The ownership modifier which should be converted to an <code>AnnotationInfo</code> instance.
   * @return an <code>AnnotationInfo</code> instance.
   */
  def om2annotationInfo(om: OwnershipModifier) : AnnotationInfo = {
  	AnnotationInfo(definitions.getClass(om.getClass.getName).tpe, Nil, Nil)   
  }
}
