/*
 *              __  ____________        ____         __    
 *             / / / /_  __/ __/ ____  / __/______ _/ /__ _
 *            / /_/ / / / _\ \  /___/ _\ \/ __/ _ `/ / _ `/
 *            \____/ /_/ /___/       /___/\__/\_,_/_/\_,_/ 
 * 
 * This file is part of an implementation of the Universe Type System for
 * Scala.
 * 
 * Copyright (C) 2007-2008  Swiss Federal Institute of Technology, Zurich
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 * 
 * 
 * $Id: UTSDefaults.scala 883 2008-02-01 18:59:56Z ms $
 */
package ch.ethz.inf.sct.uts.plugin.common

import ch.ethz.inf.sct.uts.annotation._
import ch.ethz.inf.sct.uts.plugin.common.defaults._

/**
 * Defaults for the static Universe type system checks and the runtime checks.
 * 
 * @author  Manfred Stock
 * @version $Revision: 883 $
 */
trait UTSDefaults {
  /**
   * Names of methods which should be considered as pure.
   */
	def pureMethods : Map[String,List[String]]

  /**
   * Types which are immutable, in addition to the value types <code>Int</code>, 
   * <code>Boolean</code>, etc.
   */
  def immutableTypes : List[String]
  
  /**
   * Default ownership modifier for the upper bounds.
   */
  def ubOwnershipModifier : OwnershipModifier
  
  /**
   * Default owership modifier for newly created objects and method parameters 
   * (if none has been specified explicitly) and fields/parameters/return
   * types from external (i.e. not with this plugin compiled or from Java)
   * classes. 
   */
  def defaultOwnershipModifier : OwnershipModifier
  
  /**
   * Default ownership modifier for immutable types. 
   */
  def immutableDefaultOwnershipModifier : OwnershipModifier
  
  /**
   * Default ownership modifier for an <code>object</code> which is defined somewhere 
   * inside of a class, i.e. not directly in a package.
   */
  def defaultInternalObjectOwnershipModifier : OwnershipModifier
  
  /**
   * Default ownership modifier for an <code>object</code> which is defined 
   * directly in a package.
   */
  def defaultPackageObjectOwnershipModifier : OwnershipModifier
  
  /**
   * Name of object which stores information at runtime.
   */
  def runtimeObject = "ch.ethz.inf.sct.uts.rt.UTSRuntime"
    
  /**
   * Name of object which stores information at runtime using the 
   * MultiJava implementation of the runtime library.
   */
  def mjRuntimeObject = "org.multijava.universes.rt.UniverseRuntime"
    
  /**
   * Name of the class which implements the type rules.
   */
  def typerulesImplementation = "ch.ethz.inf.sct.uts.plugin.staticcheck.rules.default.DefaultTypeRules"
}

/**
 * Wrapper object around an UTSDefaults instance which provides access to the defaults.
 * 
 * @author  Manfred Stock
 * @version $Revision: 883 $
 */
object UTSDefaults extends UTSDefaults {
  /**
   * The standard defaults to use.
   */
  val standardDefaults : UTSDefaults = new StandardDefaults
  
  /**
   * The current defaults.
   */
  private var defaults = standardDefaults
  
  /**
   * Update the current defaults. The argument is a string specifying a class which
   * should be instantiated and which is an instance of UTSDefaults.
   * @param cls Name of the class which should be used to retrieve the defaults from.
   */
  def setDefaults(cls: String) {
    if (cls != null && cls.length > 0) {
      try {
        defaults = Class.forName(cls).newInstance().asInstanceOf[UTSDefaults]
      } 
      catch {
        case e : Exception => 
          System.err.println("Warning: The specified Universe defaults class (" +cls+") could not be instantiated, using the default instead.")
          defaults = standardDefaults
      }
    }
  }
  // Wrapped method calls
  def pureMethods                            = defaults.pureMethods  
  def immutableTypes                         = defaults.immutableTypes
  def ubOwnershipModifier                    = defaults.ubOwnershipModifier
  def immutableDefaultOwnershipModifier      = defaults.immutableDefaultOwnershipModifier
  def defaultOwnershipModifier               = defaults.defaultOwnershipModifier
  def defaultInternalObjectOwnershipModifier = defaults.defaultInternalObjectOwnershipModifier
  def defaultPackageObjectOwnershipModifier  = defaults.defaultPackageObjectOwnershipModifier
  override def runtimeObject                 = defaults.runtimeObject 
  override def mjRuntimeObject               = defaults.mjRuntimeObject    
  override def typerulesImplementation       = defaults.typerulesImplementation
}