/*
 *              __  ____________        ____         __    
 *             / / / /_  __/ __/ ____  / __/______ _/ /__ _
 *            / /_/ / / / _\ \  /___/ _\ \/ __/ _ `/ / _ `/
 *            \____/ /_/ /___/       /___/\__/\_,_/_/\_,_/ 
 * 
 * This file is part of an implementation of the Universe Type System for
 * Scala.
 * 
 * Copyright (C) 2007-2008  Swiss Federal Institute of Technology, Zurich
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 * 
 * 
 * $Id: UTSLogger.scala 883 2008-02-01 18:59:56Z ms $
 */
package ch.ethz.inf.sct.uts.plugin.common

import scala.util.logging.Logged
import scala.tools.nsc._

/**
 * Simple logger for the Universe type system checker.
 * 
 * @author  Manfred Stock
 * @version $Revision: 883 $
 */
trait UTSLogger extends Logged {

  /**
   * The <code>global</code> instance from the compiler.
   */
  val compiler: Global
  
  /**
   * The current compilation unit.
   */
  private var unit: compiler.CompilationUnit = null
  
  /**
   * The indentation to use for NOTICE, DEBUG, and INFO.
   */
  private var indent = ""
     
  /**
   * Update the current compilation unit.
   */
  def setUnit(u: compiler.CompilationUnit) = unit = u
  
  /**
   * The available log levels.
   */
  object Level extends Enumeration {
    val DEBUG, INFO, NOTICE, WARN, ERROR = Value
  }
  
  /**
   * Stack representing the nesting in the tree where we currently are.
   */
  private var treeStack : List[compiler.Tree] = Nil
  
  /**
   * Enter a new tree during the compilation - to be done before checking a new subtree.
   * @param tree Tree to push on stack
   */
  def enterTree(tree: compiler.Tree) {
    treeStack = tree :: treeStack
  }
  
  /**
   * Leave the current subtree.
   */
  def leaveTree {
    treeStack = treeStack.tail
  }
  
  /**
   * The loglevel. Only messages on and above are being printed.
   */
  private var loglevel = Level.NOTICE
  
  /**
   * Set the current log level to only print messages on or above the set level. 
   * Messages on level ERROR cannot be suppressed.
   * @param level The new log level.
   */
  def setLoglevel(level: Level.Value) = loglevel = level
  
  /**
   * Increase indentation by two spaces.
   * @return this which allows nesting of calls to the logger.
   */
  def ++ : UTSLogger = { ++("  "); this }
    
  /**
   * Increase indentation by a string.
   * @param s indentation string to add.
   * @return this which allows nesting of calls to the logger.
   */
  def ++ (s: String) : UTSLogger = { indent += s; this }
    
  /**
   * Decrease indentation by two characters.
   * @return this which allows nesting of calls to the logger.
   */
  def -- : UTSLogger = { indent = indent.substring(0, indent.length - 2); this }
  
  /**
   * Log a message on the current default loglevel.
   * @param message The message to log. 
   */
  def log(message: =>String) {
    log(loglevel, message)
  }
  
  /**
   * Log a message on a certain level.
   * @param level The loglevel of the message.
   * @param message The message to log.
   */
  def log(level: Level.Value, message: =>String) {
    log(level,message,if (treeStack.isEmpty) {null} else {treeStack.head.pos})
  }
  
  /**
   * Log a message on a certain level at given position. 
   * @param level The loglevel of the message.
   * @param message The message to log.
   * @param pos Position where the error occurred.
   */
  def log(level: Level.Value, message: =>String, pos: util.Position) {
    if (level >= loglevel) {
      level match {
        case Level.DEBUG  => System.out.println(indent+message)
        case Level.INFO   => System.out.println(indent+message)
        case Level.NOTICE => System.err.println(indent+message)
        case Level.WARN   => unit.warning(pos,message)
        case Level.ERROR  => unit.error(pos,message)
      }
    }
  }
  
  /**
   * Log a message on DEBUG level.
   * @param message The message to log.
   */
  def debug(message: =>String) {
    log(Level.DEBUG, message)
  }
  
  /**
   * Log a message on INFO level.
   * @param message The message to log.
   */
  def info(message: =>String) {
    log(Level.INFO, message)
  }
  
  /**
   * Log a message on NOTICE level.
   * @param message The message to log.
   */
  def notice(message: =>String) {
    log(Level.NOTICE, message)
  }
  
  /**
   * Log a message on WARN level.
   * @param message The message to log.
   */
  def warn(message: =>String) {
    log(Level.WARN, message)
  }
  
  /**
   * Log a message on WARN level using given position.
   * @param message The message to log.
   * @param pos Position where the warning occurred.
   */
  def warn(message: =>String, pos: util.Position) {
    log(Level.WARN, message, pos)
  }
  
  /**
   * Log a message on ERROR level.
   * @param message The message to log.
   */
  def error(message: =>String) {
    log(Level.ERROR, message)
  }  
  
  /**
   * Log a message on ERROR level using given position.
   * @param message The message to log.
   * @param pos Position where the error occurred.
   */
  def error(message: =>String, pos: util.Position) {
    log(Level.ERROR, message, pos)
  }
}
