/*
 *              __  ____________        ____         __    
 *             / / / /_  __/ __/ ____  / __/______ _/ /__ _
 *            / /_/ / / / _\ \  /___/ _\ \/ __/ _ `/ / _ `/
 *            \____/ /_/ /___/       /___/\__/\_,_/_/\_,_/ 
 * 
 * This file is part of an implementation of the Universe Type System for
 * Scala.
 * 
 * Copyright (C) 2007-2008  Swiss Federal Institute of Technology, Zurich
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 * 
 * 
 * $Id: TypeAbstraction.scala 817 2008-01-25 19:44:05Z ms $
 */
package ch.ethz.inf.sct.uts.plugin.staticcheck.common

import scala.tools.nsc._
import ch.ethz.inf.sct.uts.annotation._

/**
 * Put together an abstraction from the Scala compiler's type representation.
 * 
 * @author  Manfred Stock
 * @version $Revision: 817 $
 */
trait TypeAbstraction[G <: Global] extends UniverseTypeRepresentation[G] 
                                      with UTSMethodSignature[G] 
                                      with ErrorTypes[G] 
                                      with OwnershipModifierExtender
                                      with StaticErrors[G] {
  val global: G
  import global._
  
  /**
   * Enriched variant of a <code>UType</code> list.
   * @param l The poor list.
   */
  class RichUTypeList(l: List[UType]) {
    /**
     * Check if the list contains a type with given ownership modifier.
     * @param a The annotation to look for.
     * @return if the ownership modifier was contained somewhere.
     */
    def contain(a: OwnershipModifier) : Boolean = {
      l.exists(_.contains(a))   
    }
    
    /**
     * Get invalid types from a list of types.
     * @return the invalid types from the list.
     */
    def getInvalid : List[ErroneousType] = {
      l filter {
        case et: ErroneousType => true
        case _                 => false
      } map {
        case et: ErroneousType => et
      }
    }
   
    /**
     * Get invalid types concatenated in a string.
     * @return the invalid types concatenated in a string.
     */
    def invalidString : String = {
      getInvalid.mkString("","\n","")
    }
  }    
  /**
   * Convert a simple list of <code>UType</code> to its enriched variant.
   * @param l The list to enrich.
   * @return the enriched list.
   */
  implicit def utlist2rutlist(l: List[UType]) : RichUTypeList = new RichUTypeList(l)
 
  /**
   * Enriched variant of a <code>(compiler.Tree,UType)</code> list.
   * @param l The poor list.
   */
  class RichTreeUTypeList(l: List[(Tree,UType)]) {
    /**
     * Get invalid types from a list of trees mapped to types.
     * @return the invalid types from the list.
     */
    def getInvalid : List[(Tree,ErroneousType)] = {
      l filter {
        case (t,et: ErroneousType) => true
        case _                     => false
      } map {
        case (t,et: ErroneousType) => (t,et)
      }
    }
  }
   
  /**
   * Convert a simple list of <code>(compiler.Tree,UType)</code> to its enriched variant.
   * @param l The list to enrich.
   * @return the enriched list.
   */
  implicit def treeutlist2rutlist(l: List[(Tree,UType)]) : RichTreeUTypeList = new RichTreeUTypeList(l)
}