/*
 *              __  ____________        ____         __    
 *             / / / /_  __/ __/ ____  / __/______ _/ /__ _
 *            / /_/ / / / _\ \  /___/ _\ \/ __/ _ `/ / _ `/
 *            \____/ /_/ /___/       /___/\__/\_,_/_/\_,_/ 
 * 
 * This file is part of an implementation of the Universe Type System for
 * Scala.
 * 
 * Copyright (C) 2007-2008  Swiss Federal Institute of Technology, Zurich
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 * 
 * 
 * $Id: UniverseTypeRepresentation.scala 883 2008-02-01 18:59:56Z ms $
 */
package ch.ethz.inf.sct.uts.plugin.staticcheck.common

import scala.tools.nsc._
import ch.ethz.inf.sct.uts.annotation._
import ch.ethz.inf.sct.uts.plugin.common._

/**
 * Trait which encapsulates the abstractions from the internal type representation 
 * as well as some related factory and helper methods.
 * 
 * @author  Manfred Stock
 * @version $Revision: 883 $
 */
trait UniverseTypeRepresentation[G <: Global] extends UniverseTypeRepresentationBase[G] {
  self: TypeAbstraction[G] => 
  
  import global._
  import UTSDefaults._
  import extendedType._
  
  /**
   * Internal representation for Nonvariable types, ie. u C[T].
   * @param om         Main ownership modifier of the represented type.
   * @param tpe        Underlying compiler type.
   * @param typeParams Type parameters of the type.
   */
  abstract class NType(val om: OwnershipModifier, tpe: Type, val typeParams: List[UType]) extends UType(tpe) {
    // Set all typeParams to be type parameters
    typeParams foreach {_.setAsTypeParameter}
     
    /**
     * Check the type for well-formedness.
     * @return <code>this</code> if the type was well-formed.
     */
    def check : UType
   
    /**
     * Compare this type to another <code>UType</code> instance concerning equality.
     * @param that The type object to compare <code>this</code> to.
     * @return if the types are equal.
     */
    override def =:=(that: UType) : Boolean = {
      that match {
        case nt: NType         => this.om == nt.om && ((this.tpe.underlying, nt.tpe.underlying) match {
                                    case (TypeRef(pre0,sym0,args0), TypeRef(pre1,sym1,args1)) => this.tpe =:= nt.tpe || sym0.tpe =:= sym1.tpe
                                    case (_,_)                                                => this.tpe =:= nt.tpe 
                                  }) && List.forall2(this.typeParams, nt.typeParams)((a,b) => a =:= b)
        // Assume that all ErroneousTypes are equal to an NType
        case et: ErroneousType => true
        case _                 => false
      }
    }

    /**
     * Check if the type's main ownership modifier equals <code>u</code>
     * @param u The ownership modifier to look for.
     * @return if the modifiers matched.
     */
    def contains0(u: OwnershipModifier) : Boolean = {
      om == u
    }
    
    /**
     * Check recursively (over type parameters) if the type's ownership modifier 
     * equals <code>u</code>
     * @param u The ownership modifier to look for.
     * @return if the modifiers matched somewhere.
     */
    def contains(u: OwnershipModifier) : Boolean = {
      contains0(u) || typeParams.exists(_ match {
        case nt: NType => nt.contains(u)
        case _         => false 
      })
    }
    
    /**
     * Replace ownership modifiers of the type recursively by another one
     * if the current modifier is <code>om</code>.
     * @param om Modifiers which should be replaced.
     * @param by Modifier the current ones should be replaced with.
     * @return this <code>NType</code> object with substituted modifiers.
     */
    def subst(om: OwnershipModifier, by: OwnershipModifier) : NType = {
      var tparams = typeParams.map{
        case nt: NType => nt.subst(om,by)
        case t         => t
      }
      if (om == this.om) {
        nType(by,tpe,tparams)
      }
      else {
        nType(this.om,tpe,tparams)
      }
    }
    
    /**
     * Substitute type variables using the provided mapping from type variables 
     * to <code>NType</code>s if possible.
     * @param mapping Mapping from <code>Type</code>s to <code>UType</code>s.
     */
    def subst(mapping: List[(Type, UType)]) : NType = {
      nType(om, tpe, typeParams.map{
                       case nt: NType         => nt.subst(mapping)
                       case tv: TVarID        => getTypeFromMapping(mapping,tv)
                       case et: ErroneousType => et
                     }
      )
    }
    
    /**
     * Get mapping from type variables to actual types.
     * @return the mapping
     */
    def tv2ntmap : List[(Type,UType)] = {
      tpe.getClassType.typeParams.map(_.tpe).zip(typeParams)
    }
    
    /**
     * Resolve type variables. Used to find <code>NType</code>s for type 
     * variables in the type parameter list if possible.
     * @param tpe Type where the type variables should be resolved.
     * @param ntpe Type from which they are resolved, ie. which contains mapping from variable to type.
     * @return the type where type variables have been resolved.
     */
    protected def resolveTypeVars(tpe: UType, ntpe: NType) : UType
    
    /**
     * Substitute type parameters according to the given type.
     * @param ntpe Type with instances of type parameter variables.
     * @return a reference to this.
     */
    def instantiateTypeParamsFrom(ntpe: NType) : NType = {
      nType(om,tpe,typeParams.map(resolveTypeVars(_,ntpe)))
    }

    /**
     * Viewpoint adaptation of type w.r.t. an ownership modifier.
     * @param om Ownership modifier this type is to be adapted to.
     * @return the adapted type.
     */
    def |>: (om: OwnershipModifier) : UType
    
    /**
     * Viewpoint adaptation of a type w.r.t. another type.
     * @param t  The type to adapt to <code>this</code>.
     * @return the adapted type.
     */
    def |> (t: UType) : UType
     
    /**
     * Set the ownership modifier of this type to another modifier.
     * @param om The new ownership modifier.
     */
    def setOwnershipModifier(om: OwnershipModifier) : NType = {
      nType(om,tpe,typeParams)
    }
    
    /**
     * Set the ownership modifier of this type to another modifier, and also 
     * modify the type parameters this way. 
     * @param om The new ownership modifier.
     */
    def setOwnershipModifiers(om: OwnershipModifier) : NType = {
      nType(om, tpe, typeParams.map{
                       case nt: NType => nt.setOwnershipModifiers(om)
                       case t         => t
                     }
      )
    }

    /**
     * Lift type contained in <code>this</code> (which must be a subtype of
     * type <code>that</code>) to the same type as <code>that</code>, but
     * with adapted type parameters.
     * @param that Type to whose level the type in <code>this</code> should be lifted.
     * @return the lifted type.
     */
    def liftTo(that: NType) : NType

    /**
     * Upper type bound for this type which is <code>this</code> since this is an 
     * <code>NType</code>, i.e. a non-variable type. 
	   * @return the upper bound of this type, i.e. <code>this</code>.
     */
    def ubgamma : UType = {
    	this
    }
    
    /**
     * Check if modification of the referenced object 
     * is basically allowed. Could eg. be forbidden if it is 
     * an <code>any</code> reference.
     * @return if the modification is allowed.
     */
    def modificationOK : Boolean

    /**
     * Check if a given type should get an ownership modifier. <code>Unit</code>, 
     * type variables or an already annotated type should not get one.
     * @param t The type which should be checked.
     * @return if the type may be annotated.
     */
    protected def ownershipModifierAddableOn(t: Type) : Boolean = {
      (! t.containsOwnershipAnnotation) && (! t.isTypeVariable) && (t != definitions.UnitClass.tpe) && (t match {
        case MethodType(paramTypes,resultType) => (resultType != definitions.UnitClass.tpe) && (! resultType.isOwnershipAnnotated)
        case PolyType(typeParams, resultType)  => ! resultType.isOwnershipAnnotated && ! typeParams.exists(_.tpe.isOwnershipAnnotated)
        case _                                 => true
      })
    }
    
    /**
     * Add the ownership modifiers contained in this type to the type given in 
     * the argument and return the modified type. Care must be taken to add them 'correctly' 
     * as to not get failed assertions because of 'wrong' types in later compiler phases 
     * (the Scala AST is a typed AST!). 
     * @param t The type which should get ownership modifiers.
     * @return the modified type.
     */
    def addOwnershipModifiersOn(t: Type) : Type = {
      UType(t) match {
        case nt: NType if (this.tpe <:< nt.tpe && ! (this.tpe.deconst =:= nt.tpe.deconst) && nt.tpe.matches(this.tpe)) => {
          this.liftTo(nt).addOwnershipModifiersOn(t)
        }
        case tpe => { 
          if (this.ownershipModifierAddableOn(t) && t.matches(this.tpe)) {
            val res = t match {
              case MethodType(paramTypes,resultType)   => 
                MethodType(
                    paramTypes,
                    this.addOwnershipModifiersOn(resultType)
                )
              case PolyType(typeParams, resultType)    =>
                PolyType(
                    if (this.typeParams.length == typeParams.length) {
                      List.map2(this.typeParams,typeParams)((a,b) => a match {
                        case nt: NType => if (nt.ownershipModifierAddableOn(b.tpe)) { b.updateInfo(nt.addOwnershipModifiersOn(b.tpe)) } else { b }
                        case _         => b
                      })
                    }
                    else {
                      typeParams
                    },
                    this.addOwnershipModifiersOn(resultType)
                )
              case TypeRef(pre, sym, args)             =>
                typeRef(
                    pre,
                    sym,
                    if (this.typeParams.length == args.length) {
                      List.map2(this.typeParams,args)((a,b) => a match {
                        case nt: NType => nt.addOwnershipModifiersOn(b)
                        case _         => b
                      })
                    }
                    else {
                      args
                    }
                ).withAttribute(om2annotationInfo(om))
              case AnnotatedType(attribs, tp, selfsym) =>
                this.addOwnershipModifiersOn(tp) match {
                  case AnnotatedType(attr,tpe,ssym) => AnnotatedType(attr:::attribs,tpe,selfsym)
                  case tpe                          => tpe.withAttributes(attribs).withSelfsym(selfsym)
                }
              case SingleType(pre,sym) =>
                singleType(pre,sym.updateInfo(this.addOwnershipModifiersOn(sym.tpe)))
              case tp                                  => tp.withAttribute(om2annotationInfo(om))
            }
            res
          }
          else {
            t
          }
        }
      }
    }
     
    /**
     * Add the ownership modifiers contained in this type to the type given in 
     * the argument and return the modified type if they could be added, the 
     * given type if not.
     * @param t The type which should get ownership modifiers.
     * @return the modified type.
     */
    def addOwnershipModifiersOn(t: UType) : UType = {
      t match {
        case nt: NType if this.ownershipModifierAddableOn(t.tpe) =>
          nType(
              this.om,
              tpe, 
              if (nt.typeParams.length == this.typeParams.length) { 
                List.map2(this.typeParams,nt.typeParams)((a,b) => a match {
                  case nt: NType => nt.addOwnershipModifiersOn(b)
                  case _         => b
                })
              }
              else {
                nt.typeParams
              })
        case _ => t
      }
    }
    
    /**
     * Convert an <code>NType</code> to a string representation.
     * @return the string representation.
     */
    override def toString = {
      var name : String = 
        if (tpe.typeSymbol == NoSymbol || tpe.isUnit) {
          tpe.typeSymbol.simpleName.toString
        }
        else {
          tpe.typeSymbol.fullNameString
        }
      name+ {
        if (typeParams.size > 0) { 
          typeParams.mkString("[",",","]")
        }
        else {
          ""
        }
      } +
      (if (! tpe.isUnit) " " +om.name else "")
    }
  }
  
  /**
   * Internal representation for type variables.
   * @param tpe Underlying type from the Scala compiler.
   */
  abstract class TVarID(tpe: Type) extends UType(tpe) {
    /**
     * Check the type for well-formedness.
     * @return <code>this</code> if the type was well-formed, InvalidType otherwise.
     */
    def check : UType
    
    /**
     * Type variable don't have an ownership modifier, so this returns always <code>false</code>.
     * @param u The ownership modifier to look for.
     * @return if the modifiers matched somewhere, which is never the case here.
     */
    def contains(u: OwnershipModifier) : Boolean = false
    
    /**
     * Compare <code>TVarID</code> type to another <code>UType</code> concerning equality.
     * @param that The type object to compare <code>this</code> to.
     * @return if the types are equal.
     */    
    override def =:=(that: UType) : Boolean = {
      that match {
        case tv: TVarID        => this.tpe =:= that.tpe
        // Assume all type variables are equal to ErroneousTypes
        case et: ErroneousType => true
        case _                 => false
      }
    }
    
    /**
     * Find upper type bound by going upwards using gamma. If the upper bound 
     * was not ownership annotated, set it to a default. Does not do a check for 
     * well-formedness of the upper bound to prevent infinite recursion and because 
     * it is usually not required.
     * @return the upper bound of this type variable.
     */
    def subgamma : UType = {
      tpe.ubgamma match {
        case NoType => InvalidType("No upper typebound found for "+tpe+".")
        case t      => SUType(t) match {
          case nt: NType         => if (t.isOwnershipAnnotated) { nt } else { nt.setOwnershipModifier(ubOwnershipModifier) }
          case et: ErroneousType => et
          case _                 => InvalidType("No valid upper bound found.")
        }
      }
    }
    
    /**
     * Find upper type bound by going upwards using gamma. If the upper bound 
     * was not ownership annotated, set it to a default. Does check the well-formedness of 
     * the upper bound. 
     * @return the upper bound of this type variable.
     */
    def ubgamma : UType = {
      subgamma.check
    }
    
    /**
     * Convert type variable to a string.
     * @return the string representation.
     */
    override def toString = {
      if (tpe.typeSymbol != NoSymbol) {
        tpe.typeSymbol.fullNameString
      }
      else {
        tpe.typeSymbol.toString
      }
    }
    
    /**
     * Owner-as-modifier: Check if modification of the referenced object 
     * is basically allowed. Check upper bound since this is a type variable.
     * @return if the modification is allowed.
     */
    def modificationOK : Boolean = {
      ubgamma match {
        case nt: NType => nt.modificationOK
        case _         => false
      }
    }
    
    /**
     * Add an ownership modifier annotation to a <code>UType</code>. Nothing done here, as 
     * the TVarID does not contain ownership modifiers.
     * @param t0 The <code>UType</code> which should get the annotation.
     * @return the same type which was given as argument.
     */
    def addOwnershipModifiersOn(t0: UType) : UType = t0
  }
   
  /**
   * Method for the creation of UType instances which are not checked for 
   * well-formedness. Intended for the use in <code>TVarID.subgamma</code>
   * (which is used in rule WFT-2 only) and the <code>UType</code> object.
   * Checks if the given type is a type parameter or not and creates a 
   * <code>UType</code> instance accordingly.
   * @param t Type of the compiler to encapsulate into internal type.
   * @return the created and <code>TypeOption</code>-wrapped type instance. 
   */
  protected def SUType(t: Type) : UType = {
    val tpe = t.extractType
    val res = 
      if (tpe.isTypeVariable || tpe == NoType) {
        if (tpe.isOwnershipAnnotated) {
          InvalidType("Type parameters must not be ownership annotated.")
        }
        else {
          tVarID(tpe)  
        }
      }
      else {
        def getOm(tp: Type) = {
          if (tp.ownershipModifiers.length >= 1) {
            if (tp.ownershipModifiers.length > 1) {
              logger.notice("More than one ownership modifier annotation found! Using only the first one, "+tp.ownershipModifiers.head+".")
            }
            tp.ownershipModifiers.head
          }
          else {
            // Set om to default (usually peer)
            if (tp.isImmutable) immutableDefaultOwnershipModifier else defaultOwnershipModifier
          }
        }
        // Get type parameters and ownership modifier from the type - special handling for PolyTypes, since
        // they contain the information in a slightly different way
        val (typeParams,om) = tpe match {
        case PolyType(typeParams,resultType) => 
          (typeParams map {_.tpe} map {SUType(_)}, getOm(resultType))
        case _                               => 
          (tpe.typeArgs.map(SUType(_)),            getOm(tpe)) 
        }
        nType(om, tpe, typeParams)
      }
    res
  }
   
  /**
   * Method for the creation of UType instances. Checks if the given type is a
   * type parameter and creates a <code>UType</code> instance accordingly. Also
   * checks the type for well-formedness.
   * @param t Compiler type which should be encapsulated in a <code>UType</code>.
   * @return the created and <code>TypeOption</code>-wrapped type instance. 
   */
  def UType(t: Type) : UType = {
    SUType(t).check
  }
  
  /**
   * Factory method to create new <code>NType</code> instances.
   * @param om         Main modifier of the nonvariable type.
   * @param tpe        The encapsulated type.
   * @param typeParams The type parameters.
   * @return the new <code>NType</code> instance.
   */
  def nType(om: OwnershipModifier, tpe: Type, typeParams: List[UType]) : NType
  
  /**
   * Factory method to create new <code>TVarID</code> instances.
   * @param tpe Type to encapsulate in the type variable identifier.
   */
  def tVarID(tpe: Type) : TVarID
  
  /**
   * Get a type from the mapping of type variables to types.
   * @param mapping The mapping from type variables to types.
   * @param tv      The type variable to look for.
   * @return the type which is mapped from the given type variable
   *         if found, the type variable if not.
   */
  def getTypeFromMapping(mapping: List[(Type, UType)],tv: TVarID) : UType = {
    val types = mapping.filter{case (a,b) => a == tv.tpe}
                       .map{case (a,b) => b}
    if (types.isEmpty) {
      tv
    }
    else {
      types.head
    }
  }
}
