/*
 *              __  ____________        ____         __    
 *             / / / /_  __/ __/ ____  / __/______ _/ /__ _
 *            / /_/ / / / _\ \  /___/ _\ \/ __/ _ `/ / _ `/
 *            \____/ /_/ /___/       /___/\__/\_,_/_/\_,_/ 
 * 
 * This file is part of an implementation of the Universe Type System for
 * Scala.
 * 
 * Copyright (C) 2007-2008  Swiss Federal Institute of Technology, Zurich
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 * 
 * 
 * $Id: DefaultOwnershipModifierExtender.scala 817 2008-01-25 19:44:05Z ms $
 */
package ch.ethz.inf.sct.uts.plugin.staticcheck.rules.default

import ch.ethz.inf.sct.uts.annotation._
import ch.ethz.inf.sct.uts.plugin.staticcheck.common._

/**
 * Trait to implement the actual behavior of the ownership modifers.
 * 
 * @author  Manfred Stock
 * @version $Revision: 817 $
 */
trait DefaultOwnershipModifierExtender extends OwnershipModifierExtender {
  /**
   * Implicitly convert an ownership modifier to an <code>ExtendedOwnershipModifier</code>.
   * @param om The normal ownership modifier which will be encapsulated..
   * @return the enriched ownership modifier
   */
  implicit def om2eom(om: OwnershipModifier) : ExtendedOwnershipModifier = new DefaultExtendedOwnershipModifier(om) 

  /**
   * Default implementation of the ownership modifier behavior.
   * @param om The normal ownership modifier which will be encapsulated.
   */
  class DefaultExtendedOwnershipModifier(om: OwnershipModifier) extends ExtendedOwnershipModifier(om) {
    def parent = om match {
      case any()  => null
      case some() => any()
      case lost() => some()
      case rep()  => lost()
      case peer() => lost()
      case thiz() => peer()
    }
    
    /**
     * Is this modifier a main modifier?
     * @return if the encapsulated modifier is a main modifier.
     */
    def isMainModifier = {
      om match {
        case any()  => true
        case peer() => true
        case rep()  => true
        case _      => false
      }
    }
     
    /**
     * Viewpoint adaptation of an ownership modifier w.r.t. an ownership modifier.
     * @param uprime Modifier which is to be adapted.
     * @return the adapted ownership modifier.
     */
    def |> (uprime: OwnershipModifier) : OwnershipModifier = {
      (this.om, uprime) match {
        case (peer(), peer())                => peer()
        case (rep(),  peer())                => rep()
        case (_,      some())                => some()
        case (_,      any())                 => any()
        case (thiz(), _) if uprime != thiz() => uprime
        case (_,      _)                     => lost()
      }
    }
  }
}
