/*
 *              __  ____________        ____         __    
 *             / / / /_  __/ __/ ____  / __/______ _/ /__ _
 *            / /_/ / / / _\ \  /___/ _\ \/ __/ _ `/ / _ `/
 *            \____/ /_/ /___/       /___/\__/\_,_/_/\_,_/ 
 * 
 * This file is part of an implementation of the Universe Type System for
 * Scala.
 * 
 * Copyright (C) 2007-2008  Swiss Federal Institute of Technology, Zurich
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 * 
 * 
 * $Id: DefaultUniverseTypeRepresentation.scala 817 2008-01-25 19:44:05Z ms $
 */
package ch.ethz.inf.sct.uts.plugin.staticcheck.rules.default

import scala.tools.nsc._
import ch.ethz.inf.sct.uts.annotation._
import ch.ethz.inf.sct.uts.plugin.common._
import ch.ethz.inf.sct.uts.plugin.staticcheck.common._

/**
 * Trait which encapsulates the abstractions from the internal type representation 
 * as well as some factory methods.
 * 
 * @author  Manfred Stock
 * @version $Revision: 817 $
 */
trait DefaultUniverseTypeRepresentation[G <: Global] extends UniverseTypeRepresentation[G] {
  self: DefaultTypeAbstraction[G] =>
  import global._
  import extendedType._
  
  /**
   * Factory method to create new <code>TVarID</code> instances.
   * @param tpe Type to encapsulate in the type variable identifier.
   */
  def tVarID(tpe: Type) : TVarID = new DTVarID(tpe)
  
  /**
   * Default implementation of a type variable identifier.
   * @param tpe Underlying type from the Scala compiler.
   */
  class  DTVarID(tpe: Type) extends TVarID(tpe) {
    /**
     * Viewpoint adaptation of type w.r.t. an ownership modifier.
     * @param om Ownership modifier this type is to be adapted to.
     * @return the adapted type.
     */
    def |>: (om: OwnershipModifier) : UType = this
  
    /**
     * Check the type for well-formedness.
     * @return <code>this</code> if the type was well-formed, InvalidType otherwise.
     */
    def check : UType = {
      // Check WFT-1
      if (this.tpe == NoType 
          || (this.tpe.ubgamma match {
                case NoType => false
                case _      => true
             })
      ) {
        this
      }
      else {
        InvalidType(tpe+" is a NoType or does not have an upper bound.")
      }
    }
     
    /**
     * Subtyping rule ST-4, which also includes rule ST-3 => the upper bound is also 
     * allowed to be a subtype of <code>that</code>.
     * @param that The other type.
     * @return if the this is a subtype of the other type. 
     */
    def <:<(that: UType) : Boolean = {
      // ST-4
      this =:= that || (SUType(this.tpe.gamma) <:< that)
    }
     
    /**
     * Limited covariance rule TA-3.
     * @param that The other type.
     * @return if the rule is fulfilled.
     */
    def ta(that: UType) : Boolean = {
      // TA-1
      (this =:= that) ||
        // TA-3
        (ubgamma match {
          case nt: NType  => that =:= nt.subst(peer(),some()).subst(rep(),some()).subst(any(),some())
          // Should never happen
          case tv: TVarID => logger.notice("Upper bound of "+this+" was a type variable - should never happen"); false
          case t          => logger.warn(t.toString); false
        })
    } 
  }
  
  /**
   * Factory method to creat new <code>NType</code> instances.
   * @param om         Main ownership modifier of the represented type.
   * @param tpe        Underlying compiler type.
   * @param typeParams Type parameters of the type.
   * @return the new <code>NType</code> instance.
   */
  def nType(om: OwnershipModifier, tpe: Type, typeParams: List[UType]) : NType = new DNType(om,tpe,typeParams)
  
  /**
   * Default implementation of the internal representation for Nonvariable types, ie. u C[T].
   * @param om         Main ownership modifier of the represented type.
   * @param tpe        Underlying compiler type.
   * @param typeParams Type parameters of the type.
   */
  class DNType(om: OwnershipModifier, tpe: Type, typeParams: List[UType]) extends NType(om, tpe, typeParams) {
    /**
     * Check the type for well-formedness, ie. rule WFT-2.
     * @return <code>this</code> if the type was well-formed.
     */
    def check : UType = {
      var errors = InvalidTypes(Nil)
      if (typeParams.map{_.check}.forall{
            case et: ErroneousType => errors = errors + et; false
            case t: UType          =>                       true
          }
      ) {
        // Check well-formedness rule WFT-2
        def checkSt(a: UType, b: UType) : Boolean = {
          if (a <:< b) {
            true
          }
          else {
            errors = errors + InvalidType(typeError(a,b,"Actual type argument not a subtype of the viewpoint adapted upper bound.")); false
          }
        }
        val tv2nt = tv2ntmap
        if (typeParams.forall(p => {
              val tparam = tv2nt.filter{case (a,b) => b =:= p}.map{case (a,b) => a}
              if (tparam.size >= 1) {
                UType(tparam.head) match {
                  case nt: NType         => checkSt(p, this |> nt)
                  case tv: TVarID        => checkSt(p, this |> tv.subgamma)
                  case et: ErroneousType => errors = errors + et; false
                }
              } else {
                true
              }
            })
        ) {
          if (om == some() && !isTypeParameter) {
            InvalidType("@some not allowed as main modifier: "+this)
          }
          else {
            this
          }
        }
        else {
          errors
        }
      }
      else {
        errors
      }
    }
      
    /**
     * Resolve type variables. Used to find <code>NType</code>s for type 
     * variables in the type parameter list if possible.
     * @param tpe Type where the type variables should be resolved.
     * @param ntpe Type from which they are resolved, ie. which contains mapping from variable to type.
     * @return the type where type variables have been resolved.
     */
    override protected def resolveTypeVars(tpe: UType, ntpe: NType) : UType = {
      var tv2nt = tv2ntmap
      tpe match { 
        case nt: NType         => nt.instantiateTypeParamsFrom(ntpe).subst(tv2nt)
        case tv: TVarID        => { 
          (ntpe.tpe match {
            case tr@TypeRef(pre,sym,args) => UType(tr.transform(tv.tpe))
            case _                        => UType(tv.tpe.asSeenFrom(ntpe.tpe, tv.tpe.typeSymbol.owner))
          }) match {
            case nt: NType         => nt.subst(tv2nt).subst(some(), lost())
            case tv: TVarID        => getTypeFromMapping(tv2nt,tv)
            case et: ErroneousType => et
          }
        }
        case et: ErroneousType => et
      }
    }
     
    /**
     * Viewpoint adaptation of type w.r.t. an ownership modifier.
     * @param om Ownership modifier this type is to be adapted to.
     * @return the adapted type.
     */
    def |>: (om: OwnershipModifier) : UType = {
      nType(om |> this.om,tpe,typeParams.map(om |>: _)) 
    }
       
    /**
     * Viewpoint adaptation of a type w.r.t. another type.
     * @param nt Nonvariable type to whose viewpoint <code>t</code> gets adapted.
     * @param t  The type to adapt.
     * @return the adapted type.
     */
    def |> (t: UType) : UType = {
      resolveTypeVars(this.om |>: t,this)
    }
    
    /**
     * Lift type contained in <code>this</code> (which must be a subtype of
     * type <code>that</code>) to the same type as <code>that</code>, but
     * with adapted type parameters.
     * @param that Type to whose level the type in <code>this</code> should be lifted.
     * @return the lifted type.
     */
    def liftTo(that: NType) : NType = {
      var ltype : NType = this
      // No explicit check if <code>this</code> subclasses <code>that</code>, 
      // since this was already checked by the standard Scala compiler.
      if (!(ltype.tpe sameClassAs that.tpe)) {
        ltype = nType(
            this.om,
            that.tpe.getClassType, 
            that.tpe.getClassType.typeParams.map(s => UType(s.tpe.asSeenFrom(this.tpe, s.tpe.typeSymbol.owner)))
        )
        ltype = nType(ltype.om, ltype.tpe,ltype.typeParams.map(this.om |>: _))
        ltype = ltype.subst(tv2ntmap)
      }
      ltype
    }
      
    /**
     * Subtyping rules ST-1/2, implicitly implementing ST-3 as well.
     * @param that The type which could be a supertype of this type.
     * @return if this is the case.
     */
    def <:<(that: UType) : Boolean = {
      if (this.tpe.isScalaNull || this.tpe.isScalaNothing) {
        true
      }
      else {
        that match {
          case nt: NType         => {
            def st2(thiz: NType, that: NType) : Boolean = {
              (thiz.tpe sameClassAs that.tpe) &&
                (thiz.om <:< that.om) &&
                  List.forall2(thiz.typeParams, that.typeParams)((a,b) => 
                  (a ta b)
                  // scala.Nothing is a lower bound for all type parameters, so this should always be ok
                  || a.tpe.isScalaNothing  
                  )
            }
            st2(liftTo(nt), nt)
          }
          // Assume everything to be a subtype of the erroneous type
          case et: ErroneousType => true
          case _                 => false
        }
      }
    }
    
    /**
     * Limited covariance rules TA-1 and TA-2.
     * @param that The other type.
     * @return if TA-2 matches.
     */
    def ta(that: UType) : Boolean = {
      that match {
        case nt: NType         => {
          // TA-1
          (this =:= that) || 
            // TA-2
            ((this.tpe sameClassAs nt.tpe) &&
              List(this.om,some()).exists(_ == nt.om) && 
                List.forall2(this.typeParams, nt.typeParams)((a,b) => a ta b))
        }
        case tv: TVarID        => {
          // TA-1
          (this =:= that)
        }
        case et: ErroneousType => false
      }
    }
     
    /**
     * Owner-as-modifier: Check if modification of the referenced object 
     * is basically allowed.
     * @return if the modification is allowed.
     */
    def modificationOK : Boolean = {
      List(rep(),peer(),thiz()) contains om
    }
  }
}
